/**
 * @file        GenericQueue.c
 * @brief       Generic queue (according to std::queue).
 * @since       04.06.2020
 * @copyright   Copyright 2020 dSPACE GmbH. All rights reserved.
 */

/*----------------------------------------------------------------------------*/
/* INCLUDES                                                                   */
/*----------------------------------------------------------------------------*/

#include "GenericQueue.h"
#include "GenericQueue_Cfg.h"

#if (QUEUE_DEV_ERROR_DETECT == STD_ON)
#include "Det.h"
#endif

#include <stdlib.h>
#include <string.h>


/*----------------------------------------------------------------------------*/
/* GLOBAL FUNCTIONS                                                           */
/*----------------------------------------------------------------------------*/

/**
 * @brief Remove all elements from queue.
 * @param[in,out] queue  Queue handle.
 */
void Queue_Clear(Queue_ObjectType* queue)
{
#if (QUEUE_DEV_ERROR_DETECT == STD_ON)
    if (NULL_PTR == queue) {
        Det_ReportError(QUEUE_MODULE_ID, 0xFFu, QUEUE_SID_CLEAR, QUEUE_E_INSTANCE);
        return;
    }
    if (NULL_PTR == queue->Root)
    {
        Det_ReportError(QUEUE_MODULE_ID, queue->InstanceId, QUEUE_SID_DEINIT, QUEUE_E_UNINIT);
        return;
    }
#endif

    queue->First = 0u;
    queue->End = 0u;
}

void Queue_DeInit(Queue_ObjectType* queue)
{
#if (QUEUE_DEV_ERROR_DETECT == STD_ON)
    if (NULL_PTR == queue)
    {
        Det_ReportError(QUEUE_MODULE_ID, 0xFFu, QUEUE_SID_DEINIT, QUEUE_E_INSTANCE);
        return;
    }
    if (NULL_PTR == queue->Root)
    {
        Det_ReportError(QUEUE_MODULE_ID, queue->InstanceId, QUEUE_SID_DEINIT, QUEUE_E_UNINIT);
        return;
    }
#endif

    free(queue->Root);

    queue->Root = NULL_PTR;
    queue->First = 0u;
    queue->End = 0u;
}

uint32 Queue_Front(const Queue_ObjectType* queue, void* data, uint32 length)
{
    uint32 contLength;  /**< Length of contiguously stored data in queue. */
    uint32 size;

#if (QUEUE_DEV_ERROR_DETECT == STD_ON)
    if (NULL_PTR == queue)
    {
        Det_ReportError(QUEUE_MODULE_ID, 0xFFu, QUEUE_SID_FRONT, QUEUE_E_INSTANCE);
        return 0u;
    }
    if (NULL_PTR == queue->Root)
    {
        Det_ReportError(QUEUE_MODULE_ID, queue->InstanceId, QUEUE_SID_FRONT, QUEUE_E_UNINIT);
        return 0u;
    }
    if (NULL_PTR == data)
    {
        Det_ReportError(QUEUE_MODULE_ID, queue->InstanceId, QUEUE_SID_FRONT, QUEUE_E_PARAM);
        return 0u;
    }
    if (0u == length)
    {
        Det_ReportError(QUEUE_MODULE_ID, queue->InstanceId, QUEUE_SID_FRONT, QUEUE_E_PARAM);
        return 0u;
    }
#endif

    /* Limit length to queue size. */
    size = Queue_GetSize(queue);

    if (0u == size) { return 0u; }

    if (length > size)
    {
        length = size;
    }

    /* Take data from queue */
    contLength = queue->MaxSize - queue->First;

    if (NULL_PTR != data)
    {
        if (length <= contLength)
        {
            /* Data is stored contiguous. */
            memcpy(data, queue->Root+queue->First, length);
        }
        else
        {
            /* Data is split due to queue turnaround. */
            memcpy(data, queue->Root+queue->First, contLength);
            memcpy(((uint8*)data)+contLength, queue->Root, length-contLength);
        }
    }

    return length;
}

uint32 Queue_GetAvailableSize(const Queue_ObjectType* queue)
{
#if (QUEUE_DEV_ERROR_DETECT == STD_ON)
    if (NULL_PTR == queue)
    {
        Det_ReportError(QUEUE_MODULE_ID, 0xFFu, QUEUE_SID_GET_AVAILABLE_SIZE, QUEUE_E_INSTANCE);
        return 0u;
    }
    if (NULL_PTR == queue->Root)
    {
        Det_ReportError(QUEUE_MODULE_ID, queue->InstanceId, QUEUE_SID_GET_AVAILABLE_SIZE, QUEUE_E_UNINIT);
        return 0u;
    }
#endif

    return queue->MaxSize - Queue_GetSize(queue) - 1u;
}

uint32 Queue_GetSize(const Queue_ObjectType* queue)
{
#if (QUEUE_DEV_ERROR_DETECT == STD_ON)
    if (NULL_PTR == queue)
    {
        Det_ReportError(QUEUE_MODULE_ID, 0xFFu, QUEUE_SID_GET_SIZE, QUEUE_E_INSTANCE);
        return 0u;
    }
    if (NULL_PTR == queue->Root)
    {
        Det_ReportError(QUEUE_MODULE_ID, queue->InstanceId, QUEUE_SID_GET_SIZE, QUEUE_E_UNINIT);
        return 0u;
    }
#endif

    if (queue->End < queue->First)
    {
        return queue->End + queue->MaxSize - queue->First;
    }

    return queue->End - queue->First;
}

Std_ReturnType Queue_Init(Queue_ObjectType* queue, uint32 maxSize)
{
    static uint8 instanceCnt = 0u;

#if (QUEUE_DEV_ERROR_DETECT == STD_ON)
    if (NULL_PTR == queue)
    {
        Det_ReportError(QUEUE_MODULE_ID, 0xFFu, QUEUE_SID_INIT, QUEUE_E_INSTANCE);
        return E_NOT_OK;
    }
    if (NULL_PTR != queue->Root)
    {
        Det_ReportError(QUEUE_MODULE_ID, 0xFFu, QUEUE_SID_INIT, QUEUE_E_INSTANCE);
        return E_NOT_OK;
    }
    if (0u == maxSize)
    {
        Det_ReportError(QUEUE_MODULE_ID, 0xFFu, QUEUE_SID_INIT, QUEUE_E_PARAM);
        return E_NOT_OK;
    }
#endif

    queue->InstanceId = instanceCnt++;
    queue->First = 0u;
    queue->End = 0u;
    queue->MaxSize = maxSize;
    queue->Root = (uint8*)malloc(queue->MaxSize);

    return (queue->Root == NULL_PTR ? E_NOT_OK : E_OK);
}

uint32 Queue_Pop(Queue_ObjectType* queue, uint32 length)
{
    uint32 size;

#if (QUEUE_DEV_ERROR_DETECT == STD_ON)
    if (NULL_PTR == queue)
    {
        Det_ReportError(QUEUE_MODULE_ID, 0xFFu, QUEUE_SID_POP, QUEUE_E_INSTANCE);
        return 0u;
    }
    if (NULL_PTR == queue->Root)
    {
        Det_ReportError(QUEUE_MODULE_ID, queue->InstanceId, QUEUE_SID_POP, QUEUE_E_UNINIT);
        return 0u;
    }
    if (0u == length)
    {
        Det_ReportError(QUEUE_MODULE_ID, queue->InstanceId, QUEUE_SID_POP, QUEUE_E_PARAM);
        return 0u;
    }
#endif

    /* Limit length to queue size. */
    size = Queue_GetSize(queue);

    if (0u == size) { return 0u; }

    if (length > size)
    {
        length = size;
    }

    /* Remove data from queue */
    queue->First += length;
    if (queue->First >= queue->MaxSize)
    {
        queue->First -= queue->MaxSize;
    }

    return length;
}

uint32 Queue_Pull(Queue_ObjectType* queue, void* data, uint32 length)
{
#if (QUEUE_DEV_ERROR_DETECT == STD_ON)
    if (NULL_PTR == queue)
    {
        Det_ReportError(QUEUE_MODULE_ID, 0xFFu, QUEUE_SID_PULL, QUEUE_E_INSTANCE);
        return 0u;
    }
    if (NULL_PTR == queue->Root)
    {
        Det_ReportError(QUEUE_MODULE_ID, queue->InstanceId, QUEUE_SID_PULL, QUEUE_E_UNINIT);
        return 0u;
    }
    if (0u == length)
    {
        Det_ReportError(QUEUE_MODULE_ID, queue->InstanceId, QUEUE_SID_PULL, QUEUE_E_PARAM);
        return 0u;
    }
#endif

    if (NULL_PTR != data)
    {
        length = Queue_Front(queue, data, length);
    }

    return Queue_Pop(queue, length);
}

Std_ReturnType Queue_Push(Queue_ObjectType* queue, const void* data, uint32 length)
{
    uint32 contLength;  /**< Length of contiguous available space in queue. */

#if (QUEUE_DEV_ERROR_DETECT == STD_ON)
    if (NULL_PTR == queue)
    {
        Det_ReportError(QUEUE_MODULE_ID, 0xFFu, QUEUE_SID_PUSH, QUEUE_E_INSTANCE);
        return E_NOT_OK;
    }
    if (NULL_PTR == queue->Root)
    {
        Det_ReportError(QUEUE_MODULE_ID, queue->InstanceId, QUEUE_SID_PUSH, QUEUE_E_UNINIT);
        return E_NOT_OK;
    }
    if (NULL_PTR == data)
    {
        Det_ReportError(QUEUE_MODULE_ID, queue->InstanceId, QUEUE_SID_PUSH, QUEUE_E_PARAM);
        return E_NOT_OK;
    }
    if (0u == length)
    {
        Det_ReportError(QUEUE_MODULE_ID, queue->InstanceId, QUEUE_SID_PUSH, QUEUE_E_PARAM);
        return E_NOT_OK;
    }
#endif

    /* Check for available capacity */
    if (Queue_GetSize(queue) + length >= queue->MaxSize) { return E_NOT_OK; }

    /* Copy data to queue */
    contLength = queue->MaxSize - queue->End;

    if (length <= contLength)
    {
        /* Data can be stored contiguous. */
        memcpy(queue->Root+queue->End, data, length);
    }
    else
    {
        /* Data must be split due to queue turnaround. */
        memcpy(queue->Root+queue->End, data, contLength);
        memcpy(queue->Root, ((uint8*)data)+contLength, length-contLength);
    }

    queue->End += length;
    if (queue->End >= queue->MaxSize)
    {
        queue->End -= queue->MaxSize;
    }

    return E_OK;
}

/*----------------------------------------------------------------------------*/
/* END OF FILE                                                                */
/*----------------------------------------------------------------------------*/
